/*
 * Copyright (C) 2000 - 2024 Silverpeas
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * As a special exception to the terms and conditions of version 3.0 of
 * the GPL, you may redistribute this Program in connection with Free/Libre
 * Open Source Software ("FLOSS") applications as described in Silverpeas's
 * FLOSS exception.  You should have received a copy of the text describing
 * the FLOSS exception, and it is also available here:
 * "https://www.silverpeas.org/legal/floss_exception.html"
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.silverpeas.web.socialnetwork.authentication;

import org.silverpeas.core.admin.service.AdminException;
import org.silverpeas.core.admin.user.UserRegistrationService;
import org.silverpeas.core.security.authentication.exception.AuthenticationException;
import org.silverpeas.core.security.authentication.verifier.AuthenticationUserVerifierFactory;
import org.silverpeas.core.security.authentication.verifier.UserCanLoginVerifier;
import org.silverpeas.core.socialnetwork.connectors.SocialNetworkConnector;
import org.silverpeas.core.socialnetwork.model.ExternalAccount;
import org.silverpeas.core.socialnetwork.model.SocialNetworkID;
import org.silverpeas.core.socialnetwork.service.AccessToken;
import org.silverpeas.core.socialnetwork.service.SocialNetworkAuthorizationException;
import org.silverpeas.core.socialnetwork.service.SocialNetworkService;
import org.silverpeas.core.util.URLUtil;
import org.silverpeas.kernel.logging.SilverLogger;
import org.silverpeas.core.web.authentication.credentials.RegistrationSettings;
import org.springframework.social.connect.UserProfile;

import javax.inject.Inject;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * Controller to log remote social network users into Silverpeas
 *
 * @author Ludovic BERTIN
 */
public class SocialNetworkLoginController extends HttpServlet {

  private static final long serialVersionUID = 3019716885114707069L;
  private static final String NETWORK_ID = "networkId";

  @Inject
  private UserRegistrationService userRegistrationService;
  private RegistrationSettings registrationSettings = null;

  @Override
  protected void doGet(HttpServletRequest req, HttpServletResponse resp) {
    processLogin(req, resp);
  }

  @Override
  protected void doPost(HttpServletRequest req, HttpServletResponse resp) {
    processLogin(req, resp);
  }

  @Override
  public void init() throws ServletException {
    super.init();
    registrationSettings = RegistrationSettings.getSettings();
  }

  /**
   * Process Handshake to openSession user in remote social network and then log user in Silverpeas.
   * If no account has been created yet, it is automatically generated by get user's remote social
   * network profile info.
   * @param req HTTP request
   * @param resp HTTP response
   */
  private void processLogin(HttpServletRequest req, HttpServletResponse resp) {
    String command = req.getParameter("command");
    try {
      // First step, isUserStateValid Linked authentication
      if (command == null) {
        SocialNetworkID networkId = SocialNetworkID.valueOf(req.getParameter(NETWORK_ID));
        String authenticateURL = getAuthenticateURL(networkId, req);
        resp.sendRedirect(authenticateURL);
      } // Then
      else if ("backFromSocialNetworkAuthentication".equals(command)) {
        processSocialNetworkAuthentication(req, resp);
      } // Silverpeas registration
      else if ("register".equals(command)) {
        registerUser(req, resp);
      }
    } catch (ServletException | IOException e) {
      SilverLogger.getLogger(this).error(e);
      resp.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
    }
  }

  private boolean processSocialNetworkAuthentication(final HttpServletRequest req, final HttpServletResponse resp)
      throws IOException, ServletException {
    SocialNetworkID networkId = SocialNetworkID.valueOf(req.getParameter(NETWORK_ID));
    SocialNetworkConnector connector = getSocialNetworkConnector(networkId);

    AccessToken authorizationToken = getAuthorizationToken(req, connector, networkId);
    if (authorizationToken == null) {
      resp.sendRedirect("/Login");
      return false;
    }

    // Try to retrieve a silverpeas account linked to remote social network account
    String profileId = connector.getUserProfileId(authorizationToken);
    ExternalAccount account =
        SocialNetworkService.getInstance().getExternalAccount(networkId, profileId);

    // Verify that the user can login
    if (account != null && !verify(req, resp, account)) {
      return false;
    }

    SocialNetworkService.getInstance()
        .storeAuthorizationToken(req.getSession(true), networkId, authorizationToken);

    // no Silverpeas account yet
    if (account == null) {

      // if new registration is enabled on platform, redirects user to registration
      if (registrationSettings.isUserSelfRegistrationEnabled()) {
        UserProfile profile = connector.getUserProfile(authorizationToken);
        req.setAttribute("userProfile", profile);
        req.setAttribute(NETWORK_ID, networkId);
        req.getRequestDispatcher("/admin/jsp/registerFromRemoteSocialNetwork.jsp")
            .forward(req, resp);
      } // new registration is disabled : redirect user to Login
      else {
        SocialNetworkService.getInstance().removeAuthorizationToken(req.getSession(false));
        resp.sendRedirect(URLUtil.getFullApplicationURL(req) + "/Login?ErrorCode=5");
      }
    } // Silverpeas account found, log user
    else {
      RequestDispatcher dispatcher = req.getRequestDispatcher("/AuthenticationServlet");
      dispatcher.forward(req, resp);
    }
    return true;
  }

  private boolean registerUser(final HttpServletRequest req, final HttpServletResponse resp)
      throws ServletException, IOException {
    SocialNetworkID networkId = SocialNetworkID.valueOf(req.getParameter(NETWORK_ID));
    SocialNetworkConnector socialNetworkConnector = getSocialNetworkConnector(networkId);

    String firstName = req.getParameter("firstName");
    String lastName = req.getParameter("lastName");
    String email = req.getParameter("email");
    String domainId = registrationSettings.userSelfRegistrationDomainId();

    if (!registerUser(req, networkId, socialNetworkConnector, firstName, lastName, email,
        domainId)) {
      RequestDispatcher dispatcher = req.getRequestDispatcher("/admin/jsp/alreadyRegistered.jsp");
      dispatcher.forward(req, resp);
      return false;
    }

    // Forward to authentication servlet
    RequestDispatcher dispatcher = req.getRequestDispatcher("/AuthenticationServlet");
    dispatcher.forward(req, resp);
    return true;
  }

  private boolean registerUser(final HttpServletRequest req, final SocialNetworkID networkId,
      final SocialNetworkConnector socialNetworkConnector, final String firstName,
      final String lastName, final String email, final String domainId) {
    try {
      String userId = userRegistrationService.registerUser(firstName, lastName, email, domainId);
      AccessToken authorizationToken = SocialNetworkService.getInstance().
          getStoredAuthorizationToken(req.getSession(true), networkId);
      String profileId = socialNetworkConnector.getUserProfileId(authorizationToken);
      SocialNetworkService.getInstance().createExternalAccount(networkId, userId, profileId);
    } catch (AdminException e) {
      SilverLogger.getLogger(this).error(e);
      return false;
    }
    return true;
  }

  private boolean verify(final HttpServletRequest req, final HttpServletResponse resp,
      final ExternalAccount account) throws IOException {
    UserCanLoginVerifier userStateVerifier =
        AuthenticationUserVerifierFactory.getUserCanLoginVerifier(account.getSilverpeasUserId());
    try {
      userStateVerifier.verify();
    } catch (AuthenticationException e) {
      SocialNetworkService.getInstance().removeAuthorizationToken(req.getSession(false));
      resp.sendRedirect(
          URLUtil.getFullApplicationURL(req) + userStateVerifier.getErrorDestination());
      return false;
    }
    return true;
  }

  private AccessToken getAuthorizationToken(final HttpServletRequest req,
      final SocialNetworkConnector connector, final SocialNetworkID networkId) {
    try {
      return connector.exchangeForAccessToken(req, getRedirectURL(networkId, req));
    } catch (SocialNetworkAuthorizationException e) {
      SilverLogger.getLogger(this).warn(e);
    }
    return null;
  }

  private String getRedirectURL(SocialNetworkID networkId, HttpServletRequest request) {
    StringBuilder redirectURL = new StringBuilder();
    redirectURL.append(URLUtil.getFullApplicationURL(request));
    redirectURL.append(
        "/SocialNetworkLogin?command=backFromSocialNetworkAuthentication&networkId=");
    redirectURL.append(networkId);
    return redirectURL.toString();
  }

  private SocialNetworkConnector getSocialNetworkConnector(SocialNetworkID networkId) {
    return SocialNetworkService.getInstance().getSocialNetworkConnector(networkId);
  }

  /**
   * Get URL to invoke remote social network authentication
   *
   * @return
   */
  private String getAuthenticateURL(SocialNetworkID networkId, HttpServletRequest request) {
    return getSocialNetworkConnector(networkId).buildAuthenticateUrl(getRedirectURL(networkId,
        request));
  }
}
